/**
 * 
 */
package gov.va.med.mhv.phr.decorator;

import gov.va.med.mhv.phr.transfer.ImpreciseDateComposite;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public final class DecoratorUtils  {
	
	private static final Log LOG = LogFactory.getLog(DecoratorUtils.class);

    private static final String UNKNOWN = StringEscapeUtils.
        escapeHtml("Unknown");

    private static final String BLANK = StringEscapeUtils.escapeHtml("");
    
    private static ThreadLocal<DateFormat> dateTimeFormat = new ThreadLocal<DateFormat>() {
		@Override
		protected synchronized  DateFormat initialValue() {
	    	return new SimpleDateFormat("MM/dd/yyyy HH:mm");
		}
    };
    
    public static DateFormat getDateTimeFormat() {
    	return dateTimeFormat.get();
    }

    private static ThreadLocal<DateFormat> dateFormat = new ThreadLocal<DateFormat>() {
		@Override
		protected synchronized  DateFormat initialValue() {
	    	return new SimpleDateFormat("MM/dd/yyyy");
		}
    };
    
    public static DateFormat getDateFormat() {
    	return dateFormat.get();
    }

    private static ThreadLocal<DateFormat> timeFormat = new ThreadLocal<DateFormat>() {
		@Override
		protected synchronized  DateFormat initialValue() {
	    	return new SimpleDateFormat("HH:mm");
		}
    };
    
    public static DateFormat getTimeFormat() {
    	return timeFormat.get();
    }
    
    private static ThreadLocal<DateFormat> impreciseDateFormat = new ThreadLocal<DateFormat>() {
		@Override
		protected synchronized  DateFormat initialValue() {
			MultiPatternDateFormat result = new MultiPatternDateFormat(new SimpleDateFormat("MM/dd/yyyy HH:mm"));
	       	// Add possible date formats for data from the integration framework here
	    	result.addFormat("yyyyMMdd.HHmmss");
	    	return result;
		}
    };
    
    public static DateFormat getImpreciseDateFormat() {
    	return impreciseDateFormat.get();
    }

    public static String escapedValueOrUnknown(String value) {
        return StringUtils.isBlank(value) ? UNKNOWN : escapedValue(value);
    }

    public static String escapedValueOrBlank(String value) {
        return StringUtils.isBlank(value) ? BLANK : escapedValue(value);
    }

    public static String escapedValue(String value) {
        return StringEscapeUtils.escapeHtml(value);
    }

    public static String formatDateTime(Date value) {
        return (value != null) ? getDateTimeFormat().format(value) : null;
    }

    public static String formatDate(Date value) {
        return (value != null) ? getDateFormat().format(value) : null;
    }

    public static String formatTime(Date value) {
        return (value != null) ? getTimeFormat().format(value) : null;
    }

    public static String formatDateTime(ImpreciseDateComposite dateComposite) {
        if (dateComposite == null) {
            return null;
        }
        Date date = toDate(dateComposite);
        if ( date != null) {
        	return formatDateTime(date);
        } else {
        	return dateComposite.getImpreciseValue();
        }
    }

    public static String formatDate(ImpreciseDateComposite dateComposite) {
        if (dateComposite == null) {
            return null;
        }
        Date date = toDate(dateComposite);
        if ( date != null) {
        	return formatDate(date);
        } else {
        	return dateComposite.getImpreciseValue();
        }
    }
    
    /**
     * This method attempts to get the value and tries to extract the value depending on the imprecise date 
     * (which could be a string) or precise value (which could be a timestamp).
     * @param dateComposite
     * @return
     */    
    
    public static String extractValue(ImpreciseDateComposite dateComposite) {
        if (dateComposite == null) {
            return null;
        }
        String impreciseValue = dateComposite.getImpreciseValue();
        if(impreciseValue != null && !"".equals(impreciseValue)){
        	return impreciseValue;
        } 
        return formatDate(new Date(dateComposite.getValue().getTime()));
    }
    
    
    
    /**
     * This method attempts to get a Date from the date composite by
     * first parsing the impreciseValue string.  If that doesn't work the
     * precise value is returned.
     * @param dateComposite
     * @return
     */
    public static Date toDate(ImpreciseDateComposite dateComposite) {
    	if (dateComposite == null) {
    		return null;
    	}
    	String impreciseValue = dateComposite.getImpreciseValue();
    	LOG.debug("Imprecise value: " + impreciseValue);
    	if ( impreciseValue != null ) {
    		try {
    			return getImpreciseDateFormat().parse(impreciseValue);
    		} catch (ParseException e) {
    			LOG.info("Exception parsing '" + impreciseValue + "' using " + dateComposite.getValue(), e);
    			return new Date(dateComposite.getValue().getTime());
    		}
    	} else {
    		return new Date(dateComposite.getValue().getTime());
    	}
    }
    
    public static Date removeTime(Date date) {
        if(date == null) {
          throw new IllegalArgumentException("The argument 'date' cannot be null.");
        }

        Calendar calendar = Calendar.getInstance();
        calendar.setLenient(false);
        calendar.setTime(date);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        
        return calendar.getTime();

      }
    

}
